/***************************************************************************
  This source code is part of the Cognachrome 2000 vision system.
  Copyright 1994, Anne Wright, anarch@mit.edu,
  Randy Sargent, rsargent@mit.edu
  This copy is not to be distributed in any form
  and is to be used only with express permission of the author.
****************************************************************************/

#include "332.h"
#include "vision.h"
#include "scampvis.pro"

int run_on_reset=1;
int startup_stack= 2048;

int s_roll=0,s_pitch=0,s_yaw=0;
int s_x=0,s_y=0,s_z=0;

int s_count=0,s_center_ok=0,s_secondary_ok=0,s_y_ok=0,s_x_ok=0;
void main()
{
  char c;
  int i;

  init_vision();
  scamp_init();

  visui_handle_char('\n');
  while(!exit_vision){
     fclear(stdin);
     fclear(vision_stream);
     fclear(irstream);

     c=getchar();
     switch(c){
     case '?':
       visui_handle_char('?');
       printf("\nScamp commands:\n");
       printf("\t0: Frob Scamp\n");
       printf("\t1: Change Scamp enables\n");
       printf("\t2: Scamp seek channel 1 simple\n");
       printf("\t3: Scamp seek channel 1 PD\n");
       printf("\t4: Scamp seek target\n");
       printf("\t5: Upload frames\n");
       printf("Scamp persistents:\n");
       printf("\tDEBUG: scamp_viewser=0, scamp_pd_debug=0, scamp_targ_dbg\n");
       printf("\tENABLES: enable_roll, enable_pitch, enable_yaw, enable_x_trans\n");
       printf("\t\tenable_y_trans,enable_z_trans\n");
       printf("\tTRACK: center_row=125, center_col=125\n");
       printf("\tSEEK: z_gain=1000, yaw_gain=1000, y_gain=1000, pitch_gain=1000\n");
       printf("\t\tx_thrust=60\n");
       printf("\tPD: colvel_factor=5000, colvel_decay=700\n");
       break;
     case '0':
       scamp_frob();
       break;
     case '1':
       scamp_enable_settings();
       break;
     case '2':
       do_vision(visui_vs,scamp_seek_ch1_simple_frame_done);
       break;
     case '3':
       scamp_seek_init();
       do_vision(visui_vs,scamp_seek_ch1_pd_frame_done);
       break;
     case '4':
       /* Reset benchmark counters */
       s_count=0;
       s_center_ok=0;
       s_secondary_ok=0;
       s_y_ok=0;
       s_x_ok=0;

       do_vision3(visui_vs,scamp_seek_target_frame_done);
       set_video_mux(0);	/* show channel a at the end */

       /* print stats */
       printf("Total %d, center_ok %d (%d%%), sec_ok %d (%d%%), y_ok %d (%d%%), x_ok %d (%d%%)\n",
	      s_count,
	      s_center_ok,(s_center_ok*100)/s_count,
	      s_secondary_ok,(s_secondary_ok*100)/s_count,
	      s_y_ok,(s_y_ok*100)/s_count,
	      s_x_ok,(s_x_ok*100)/s_count);

       break;
     case '5':
       i=0;
       while(!hit_q_stop_vision()) {
	 printf("Frame %d\n",i++);
	 digitize_frame24(visui_fb);
	 plot_frame24(visui_fb,"rgb");
       }
       break;
     default:
       visui_handle_char(c);
       continue;
     }
     visui_handle_char('\n');
  }
  visui_term();
}

Persistent scamp_viewser;
Persistent scamp_pd_debug;
Persistent scamp_targ_dbg;
Persistent enable_roll;
Persistent enable_pitch;
Persistent enable_yaw;
Persistent enable_x_trans;
Persistent enable_y_trans;
Persistent enable_z_trans;
Persistent center_row;
Persistent center_col;

Persistent colvel_factor;
Persistent colvel_decay;

Persistent   pitch_gain;
Persistent   yaw_gain;
Persistent   z_gain;
Persistent   x_gain;
Persistent   x_goal;
Persistent   y_gain;

Persistent   area_min;
Persistent   x_thrust;

FILE *scamp_out;

/* defaults updated after 3/19/96 with SCAMP (first day) */
void scamp_init()
{
  scamp_viewser=init_persistent("scamp_viewser",0);
  scamp_pd_debug=init_persistent("scamp_pd_debug",0);
  scamp_targ_dbg=init_persistent("scamp_targ_dbg",0);

  enable_roll=init_persistent("enable_roll",0);
  enable_pitch=init_persistent("enable_pitch",0);
  enable_yaw=init_persistent("enable_yaw",1);
  
  enable_x_trans=init_persistent("enable_x_trans",1);
  enable_y_trans=init_persistent("enable_y_trans",1);
  enable_z_trans=init_persistent("enable_z_trans",1);

  center_row=init_persistent("center_row",125);
  center_col=init_persistent("center_col",125);

  pitch_gain=init_persistent("pitch_gain",1000);
  z_gain=init_persistent("z_gain",6000);
  x_gain=init_persistent("x_gain",1000);
  x_goal=init_persistent("x_goal",150);
  y_gain=init_persistent("y_gain",1000);
  yaw_gain=init_persistent("yaw_gain",3000);

  x_thrust=init_persistent("x_thrust",120);

  colvel_factor= init_persistent("colvel_factor", 5000);
  colvel_decay= init_persistent("colvel_decay", 700);

  area_min=init_persistent("area_min",10);
  scamp_out=fopen("/serialb/scamp","RAX");
}

  
void scamp_putc(int c)
{
  c &= 0xff;
  if (*scamp_viewser) printf("[%02x]", c);
  fputc(c,scamp_out);
}

void scamp_flush(void)
{
  if (*scamp_viewser) printf("F \n");
  fflush(scamp_out);
}

void rotate_scamp(int roll, int pitch, int yaw)
{
  if (!*enable_roll) roll= 0;
  if (!*enable_pitch) pitch= 0;
  if (!*enable_yaw) yaw= 0;
  
  if (roll< -128) roll= -128;
  if (roll> 127) roll= 127;

  if (pitch< -128) pitch= -128;
  if (pitch> 127) pitch= 127;

  if (yaw< -128) yaw= -128;
  if (yaw> 127) yaw= 127;
  
  scamp_putc(0x7c);
  scamp_putc(0x02);
  scamp_putc(roll);
  scamp_putc(pitch);
  scamp_putc(yaw);
  scamp_flush();  /* Block until transmission is complete */
}

void translate_scamp(int x, int y, int z)
{
  if (!*enable_x_trans) x= 0;
  if (!*enable_y_trans) y= 0;
  if (!*enable_z_trans) z= 0;
  
  if (x< -128) x= -128;
  if (x> 127) x= 127;

  if (y< -128) y= -128;
  if (y> 127) y= 127;

  if (z< -128) z= -128;
  if (z> 127) z= 127;
  
  scamp_putc(0x74);
  scamp_putc(0x02);
  scamp_putc(x);
  scamp_putc(y);
  scamp_putc(z);
  scamp_flush();  /* Block until transmission is complete */
}

/*******************************************************************/
void scamp_frob()
{
  int c;
  char *keys= "1qaz2wsx3edc5tgb6yhn7ujm";
   
  printf("\nInstructions:\n");
  printf("                   Roll    Pitch        Yaw \n");
  printf("  Big increase:     1        2           3\n");
  printf("Small increase:     Q        W           E\n");
  printf("Small decrease:     A        S           D\n");    
  printf("  Big decrease:     Z        X           C\n");
  printf("\n");
  printf("                    x        y           z\n");
  printf("  Big increase:     5        6           7\n");
  printf("Small increase:     T        Y           U\n");
  printf("Small decrease:     G        H           J\n");    
  printf("  Big decrease:     B        N           M\n");
  printf("\n");
  printf("Press RETURN to exit mode\n");
  printf("\n");
    
  while (1) {
    printf("\rr=%+4d p=%+4d y=%+4d  |  x=%+4d y=%+4d z=%+4d    ",
	   s_roll,s_pitch,s_yaw,s_x,s_y,s_z);
     
    c= getchar();
    if (strchr(keys, c)) {
      int i= strchr(keys,c)-keys;
      int which= i/3;
      int deltas[4]={10,1,-1,-10};
      int delta= deltas[i%3];
      switch (which) {
      case 0:		/* roll */
	s_roll+=delta;
	break;
      case 1:		/* pitch */
	s_pitch+=delta;
	break;
      case 2:		/* yaw */
	s_yaw+=delta;
	break;
      case 3:		/* x */
	s_x+=delta;
	break;
      case 4:		/* y */
	s_x+=delta;
	break;
      case 5:		/* z */
	s_x+=delta; 
	break;
      }
      if (fdepth(stdin) == 0) {
	rotate_scamp(s_roll,s_pitch,s_yaw);
	translate_scamp(s_roll,s_pitch,s_yaw);	   
      }
    } else {
      switch (c) {
      case '\n':
      case '\r':
	printf("\n");
	rotate_scamp(0,0,0);
	translate_scamp(0,0,0);	
	return;
      default:
      }
    }
  }
}

void scamp_enable_settings()
{
  int val;

  val=*scamp_viewser;
  printf("\nScamp_viewser [%d]: ",val);
  scanf("%d",&val);
  *scamp_viewser=val;

  val=*enable_roll;
  printf("enable_roll [%d]: ",val);
  scanf("%d",&val);
  *enable_roll=val;

  val=*enable_pitch;
  printf("enable_pitch [%d]: ",val);
  scanf("%d",&val);
  *enable_pitch=val;

  val=*enable_yaw;
  printf("enable_yaw [%d]: ",val);
  scanf("%d",&val);
  *enable_yaw=val;

  val=*enable_x_trans;
  printf("enable_x_trans [%d]: ",val);
  scanf("%d",&val);
  *enable_x_trans=val;

  val=*enable_y_trans;
  printf("enable_y_trans [%d]: ",val);
  scanf("%d",&val);
  *enable_y_trans=val;

  val=*enable_z_trans;
  printf("enable_z_trans [%d]: ",val);
  scanf("%d",&val);
  *enable_z_trans=val;

}

int colvel;
int in_a_row;
int rerr,cerr;

void scamp_seek_init()
{
  colvel= in_a_row= 0;
}

void colvel_update(int col)
{
  static int oldcol;
  
  if (col < 0) {
    /* didn't find it */
    in_a_row= 0;
    colvel= 0;
  }
  else {
    if (in_a_row == 1) {
      colvel = 1000 * (col - oldcol);
    } else if (in_a_row > 1) {
      colvel = (*colvel_decay * colvel) / 1000 +
	(1000 - *colvel_decay) * (col - oldcol);
    }
    in_a_row++;
    oldcol= col;
  }
}

void scamp_pd_seek(int row, int col)
{ 
  int corrected_col;
  if (row < 0) {
    /* didn't find blob */
    colvel_update(-1);
    scamp_simple_seek(-1, -1);
  }
  else {
    colvel_update(col);
    corrected_col = col + (*colvel_factor) * colvel / 1000000;
    if (*scamp_pd_debug) {
      printf("R,C=%d,%d after pd C=%d\n",
	     row, col, corrected_col);
    }
    if (corrected_col < 0) corrected_col= 0;
    if (corrected_col > 255) corrected_col= 255;
    scamp_simple_seek(row, corrected_col);
  }
}

void update_scamp()
{     
  rotate_scamp(0, 0, s_yaw);
  translate_scamp(s_x, s_y, s_z);
}
       
void scamp_simple_seek(int row, int col)
{
  if (row < 0) {
    /* Didn't find the blob */
    s_yaw= 0;
    s_z= 0;
  }
  else {
    rerr=row-*center_row;
    cerr=col-*center_col;
    
    s_z=rerr*(*z_gain/1000);
    s_yaw=cerr*(*yaw_gain/1000);
  }
}

int target_in_middle()
{
  return (rerr<50 && rerr>-50) && (cerr<40 && cerr>-40);
}

int scamp_seek_ch1_simple_frame_done(Vstate *vs,Edge_buffer *eb)
{
   Blob *l;
   int row,col;

   s_y= 0;
   blobs_calculate_area(eb);
   l=blobs_select_largest(eb);
   if(l && (l->area > (*area_min))) {
     blob_find_cg(l);
     row=(int)l->ycg;
     col=(int)l->xcg;
     set_crosshairs(row,col);
     scamp_simple_seek(row, col);

     if (target_in_middle()) {
       /* Go forward */
       s_x=*x_thrust;
     } else {
       s_x= 0;
     }
   }
   else {
     scamp_simple_seek(-1, -1);
     s_x= 0;
   }
   update_scamp();
   sleep(20);
   return(!hit_q_stop_vision());
}

int scamp_seek_ch1_pd_frame_done(Vstate *vs,Edge_buffer *eb)
{
   Blob *l;
   int row,col;
   
   blobs_calculate_area(eb);
   l=blobs_select_largest(eb);
   if (l && l->area>(*area_min)) {
     blob_find_cg(l);
     row=(int)l->ycg;
     col=(int)l->xcg;
     set_crosshairs(row,col);

     scamp_pd_seek(row, col);

     if (target_in_middle()) {
       /* Go forward */
       s_x=*x_thrust;
     } else {
       s_x= 0;
     }
   } else {
     scamp_pd_seek(-1, -1);
     s_x= 0;
   }
   update_scamp();
   sleep(20);
   return(!hit_q_stop_vision());
}

/************************/
/*** FULL TARGET SEEK ***/
/************************/

int scamp_seek_target_frame_done(Vstate *vs,Edge_buffer *eb)
{
   Blob *l;
   Blob *s[4];
   Blob *left_blob, *right_blob;
   int sec_count;
   int row=-1,col=-1;
   int center_ok= 0;
   int secondary_ok= 0;
   int row_s0, row_s1, col_s0, col_s1;
   int y_err=0, left_dist=0, right_dist=0;
   static int muxval=0;
   int i;

   /* Increment counter */
   s_count++;
   
   /* Swap the output mux to see both */
   muxval=!muxval;
   set_video_mux(muxval);

   /***** DO PD SEEK TO KEEP MIDDLE TARGET IN MIDDLE */
   blobs_calculate_area(frame_eb(&vis_a));
   l=blobs_select_largest(frame_eb(&vis_a));

   if (l && l->area>(*area_min)) {
     center_ok = 1;

     /* increment center_ok counter */
     s_center_ok++;
     
     blob_find_cg(l);
     row=(int)l->ycg;
     col=(int)l->xcg;
     set_crosshairs(row,col);
     scamp_pd_seek(row, col);
     if (*scamp_targ_dbg) printf("Center; ");
   } else {
     scamp_pd_seek(-1, -1);
     if (*scamp_targ_dbg) printf("Missing center; ");
   }

   /***** FIND TWO SECONDARY BEACONS */
   if (center_ok) {
     blobs_calculate_area(frame_eb(&vis_b));
     left_blob= NULL;
     right_blob= NULL;
     sec_count= blobs_select_largest_n(s, frame_eb(&vis_b), 4);
     for (i= 0; i< sec_count; i++) {
       if (left_blob && right_blob) break;
       if (s[i]->area > *area_min) {
	 blob_find_cg(s[i]);
	 if (s[i]->xcg < col && !left_blob) left_blob= s[i];
	 if (s[i]->xcg > col && !right_blob) right_blob= s[i];
       }
     }
     if (left_blob && right_blob) {
       
       /* TODO: rotate coordinate frame until average row (of 3
	  beacons) is equal before doing the following math.  The
	  following only works because SCAMP doesn't roll or pitch */
       
       /* increment secondary_ok counter */
       s_secondary_ok++;
       
       row_s0= left_blob->ycg;
       row_s1= right_blob->ycg;
       col_s0= left_blob->xcg;
       col_s1= right_blob->xcg;
      if (*scamp_targ_dbg) printf("SEC n,r,c %ld,%d,%d n,r,c %ld,%d,%d; ",
				  left_blob->area,row_s0, col_s0,
				  right_blob->area,row_s1, col_s1);
       left_dist= col-col_s0;
       right_dist= col_s1-col;
       if (left_dist > 0 && right_dist > 0) {
	 secondary_ok= 1;
	 /* y_err is positive when we need to move to the right */
	 /* _extremely_ roughly this is in degrees off center */
	 /* TODO: this should divide by left+right TWICE, not once */
	 y_err= 45*1000*(left_dist-right_dist)/(left_dist+right_dist);
	 if (*scamp_targ_dbg) printf("Y_ERR: %d ", y_err);
	 s_y= y_err * (*y_gain) / 1000000;

	 /* increment y_ok counter */
	 s_y_ok++;
       }
     }
     else if(*scamp_targ_dbg){
       if(!sec_count) {
	 printf("SEC none; ");
       }
       else if(!s[1]) {
	 printf("SEC n,r,c %ld,%d,%d none; ",
		s[0]->area,s[0]->ycg, s[0]->xcg);
       }
       else {
	 printf("SEC n,r,c %ld,%d,%d n,r,c %ld,%d,%d too small; ",
		s[0]->area,s[0]->ycg, s[0]->xcg,
		s[1]->area,s[1]->ycg, s[1]->xcg);
       }
     }
   }
   if (!secondary_ok) s_y= 0;

   if (center_ok && secondary_ok && target_in_middle() &&
       (y_err < 5000 && y_err > -5000)) {
     int x_err;
     x_err= *x_goal - (left_dist + right_dist);
     if (*scamp_targ_dbg) printf("XERR=%d ", x_err);
     s_x= x_err * (*x_gain) / 1000;

     /* increment x ok counter */
     s_x_ok++;
   }
   else {
     s_x= 0;
   }

   if (*scamp_targ_dbg) printf("\n");
   
   update_scamp();
   sleep(20);
   return(!hit_q_stop_vision());
}

/* TODO:
   go forward if middle beacon is tiny and don't have data on other
   beacons */

