/***************************************************************************
  This source code is part of the Cognachrome 2000 vision system.
  Copyright 1994, Anne Wright, anarch@mit.edu,
  Randy Sargent, rsargent@mit.edu
  This copy is not to be distributed in any form
  and is to be used only with express permission of the author.
****************************************************************************/

/* This program was written to point a camera at a bright yellow Tang
   fish named zippy (hence the name).  The camera was mounted on a pan
   and tilt servo, which were adjusted to keep zippy in the center of
   the image. */

#include <332.h>
#include "vision.h"
#include "servo.h"
#include "zippyfishvis.pro"

int run_on_reset=1;			/* start up automatically */
int startup_stack= 2048;		/* set the stack to 2K */

Persistent center_row;			/* The target vertical
					   position of the fish in the
					   image*/
Persistent center_col;			/* The target horizontal
					   position of the fish in the
					   image*/
Persistent row_gain;			/* The amount to change the
					   tilt position by at each
					   step if it is too far off */
Persistent col_gain;			/* The amount to change the
					   pan position by if it is
					   too far off */
Persistent area_min;			/* The minimum size of the
					   blob to be considered as
					   possibly the fish */

int servo_tilt;				/* The desired position of the
					   tilt servo */
int servo_pan;				/* The desired position of the
					   pan servo */

void main()
{
  char c;
  int custom_char;
  
  /* Initialize the standard vision system hardware and software.
     (This is not actually required because it will be run
     automatically by the library initialization, but doesn't hurt) */
  init_vision();

  /* Initialize application specific stuff */
  zippyfish_init();

  /* This just prints the initial prompt */
  visui_handle_char('\n');
  
  while(!exit_vision){
    /* This clears out the serial input stream -- it may have had data
       left in them from the last operation which would generate
       spurious menu selections */
    fclear(stdin);

    /* Record whether a custom character was used or not -- defult to
       true, and set this to false in the default handler below */
    custom_char=1;
    
    /* Get a single-letter command from the serial input stream */
    c=getchar();
    switch(c){
    case '?':
      /* Print application specific help, then print the default help */
      printf("\tZippy Fish commands:\n");
      printf("\t1: Follow fish\n");
      printf("\t2: Frob servos\n");
      printf("Persistents:\n");
      printf("center_row, center_col:  Center of camera image (def 125)\n");
      printf("row_gain, col_gain:  Gains for servos\n");
      printf("area_min\n");
      visui_handle_char('?');
      break;
    case '1':
      /* This command sets both servos to the "center" location, then
	 calls the vision tracking to make the camera follow the
	 tracked object.  visui_vs is the Vstate pointer of the
	 currently selected channel.  It will default to A, but can be
	 set by the TPUVIS 'a' command */
      servo_tilt= servo_pan= 900;
      set_servos();
      do_vision(visui_vs,zippyfish_frame_done);
      break;
    case '2':
      /* This is a debugging function for adjusting the servos */
      try_servos();
      break;
    default:
      /* The character was not one of the ones which are overridden by
	 this program, so pass it off to the default TPUVIS handler */
      visui_handle_char(c);
      custom_char=0;			/* record that the default
					   handler was used */
    }
    /* This is cosmetic -- it prints a prompt when the command is
       done if it did not go to the default handler */
    visui_handle_char('\n');
  }
}

void try_servos()
{
  printf("Type tilt pan (with spaces)\n");
  scanf("%d %d", &servo_tilt, &servo_pan);
  set_servos();
  printf("Servos set to %d %d\n", servo_tilt, servo_pan);
}

void set_servos()
{
  if (servo_tilt < 0) servo_tilt= 0;
  if (servo_tilt > 1800) servo_tilt= 1800;
  if (servo_pan < 0) servo_pan= 0;
  if (servo_pan > 1800) servo_pan= 1800;

  /* set the servo values here */
  set_servo_angle(11,servo_tilt);
  set_servo_angle(15,servo_pan);
}

void zippyfish_init()
{
  /* initialize the persistents */
  center_row=init_persistent("center_row",125);
  center_col=init_persistent("center_col",125);
  row_gain=init_persistent("row_gain", 1000);
  col_gain=init_persistent("col_gain", 1000);
  area_min= init_persistent("area_min", 10);

  /* init the servos here */
  servo_init_module(tcr2);
  servo_init(11,900);
  servo_init(15,900);
}

/* This is the callback function.  It tracks a single object, checks
   whether it is large enough to bother tracking, and points the
   camera at the blob if it is. */
int zippyfish_frame_done(Vstate *vs,Edge_buffer *eb)
{
   Blob *l;
   int row,col;
   int rerr,cerr;

   /* Calculate the area of all the blobs */
   blobs_calculate_area(eb);

   /* Get the pointer to the largest blob (if any) */
   l=blobs_select_largest(eb);

   /* Check if there was a largest blob */
   if(l!=NULL) {
     /* Check that the area is greater than the minimum, assume it's
	really the fish.  If so, point the camera at it.  If not,
	ignore it. */
     if(l->area>(*area_min)) {
       /* It's large enough, so find the location of the fish in the image */
       blob_find_cg(l);

       row=(int)l->ycg;
       col=(int)l->xcg;

       /* Find the error between where the fish is in the image and
	  where we want it to be */
       rerr=row-*center_row;
       cerr=col-*center_col;

       /* Put the crosshairs on the fish (for debugging) */
       set_crosshairs(row,col);

       /* If the row position is off by more than 3, tilt the camera
	  to a different position that's closer */
       if (rerr > 3) servo_tilt += *row_gain;
       if (rerr < -3) servo_tilt -= *row_gain;

       /* If the column position is off by more than 3, pan the
	  camera to a different position that's closer */
       if (cerr > 3) servo_pan += *col_gain;
       if (cerr < -3) servo_pan -= *col_gain;
       
       /* Tell the servos to go to the (potentially) new position */
       set_servos();
     }
   }
   return(!hit_q_stop_vision());
}
